"use client";

import { useEffect, useState } from "react";
import AwardCard from "@/components/AwardCard";
import { Swiper, SwiperSlide } from "swiper/react";
import { Navigation } from "swiper/modules";
import { FaArrowLeft, FaArrowRight } from "react-icons/fa6";
import { motion, AnimatePresence } from "framer-motion";
import "swiper/css";
import "swiper/css/navigation";

type Award = {
  title: string;
  description: string;
  year: string;
  image_url: string;
};

export default function AwardsFilter({
  years,
  awards,
}: {
  years: number[];
  awards: Award[];
}) {
  const [selectedYear, setSelectedYear] = useState<number | "all">("all");
  const [isClient, setIsClient] = useState(false);

  useEffect(() => {
    setIsClient(true); // prevent SSR hydration issues
  }, []);

  const filteredAwards =
    selectedYear === "all"
      ? awards
      : awards.filter((a) => parseInt(a.year) === selectedYear);

  return (
    <>
      {/* Year Filter Slider */}
      <div className="relative flex flex-row items-center mb-[50px] justify-between gap-[10px] md:gap-[20px]">
        {/* LEFT FIXED "ALL" BUTTON */}
        <div className="flex-shrink-0">
          <button
            onClick={() => setSelectedYear("all")}
            className={`px-5 md:px-10 py-2 rounded-full text-base md:text-lg font-cabin font-medium transition ${
              selectedYear === "all"
                ? "bg-[#0c0c0c] text-white"
                : "bg-[#EEEEEE] text-[#0c0c0c] hover:bg-[#dcdcdc]"
            }`}
            style={{ fontFamily: "Cabin" }}
          >
            All
          </button>
        </div>

        {/* SCROLLING YEAR BUTTON SLIDER */}
        <div className="relative w-full md:w-[90%] overflow-hidden">
          <div className="pointer-events-none absolute right-0 top-0 h-full w-12 bg-gradient-to-l from-white to-transparent z-10"></div>

          {isClient && (
            <Swiper
              modules={[Navigation]}
              navigation={{
                nextEl: ".filter-next",
                prevEl: ".filter-prev",
              }}
              loop={true}
              slidesPerView="auto"
              spaceBetween={20}
              grabCursor={true}
              breakpoints={{
                0: { spaceBetween: 10 },
                640: { spaceBetween: 12 },
                1024: { spaceBetween: 20 },
              }}
              className="w-full flex items-center"
            >
              {years.map((year) => (
                <SwiperSlide key={year} className="!w-auto">
                  <button
                    onClick={() => setSelectedYear(year)}
                    className={`px-5 md:px-10 py-2 rounded-full text-base md:text-lg font-cabin font-medium transition ${
                      selectedYear === year
                        ? "bg-[#0c0c0c] text-white"
                        : "bg-[#EEEEEE] text-[#0c0c0c] hover:bg-[#dcdcdc]"
                    }`}
                    style={{ fontFamily: "Cabin" }}
                  >
                    {year}
                  </button>
                </SwiperSlide>
              ))}
            </Swiper>
          )}
        </div>

        {/* Navigation Arrows */}
        <div className="hidden md:flex gap-4 justify-center md:justify-end">
          <button className="filter-prev px-3 py-3 bg-[#ac9242] text-white rounded-full hover:bg-[#8a7534] transition">
            <FaArrowLeft size={20} />
          </button>
          <button className="filter-next px-3 py-3 bg-[#ac9242] text-white rounded-full hover:bg-[#8a7534] transition">
            <FaArrowRight size={20} />
          </button>
        </div>
      </div>

      {/* Animated Awards Grid */}
      <div className="grid gap-x-5 gap-y-10 sm:grid-cols-1 md:grid-cols-2 lg:grid-cols-4">
        <AnimatePresence mode="wait">
          {filteredAwards.length > 0 ? (
            filteredAwards.map((item, index) => (
              <motion.div
                key={item.title + index}
                initial={{ opacity: 0, y: 30 }}
                animate={{ opacity: 1, y: 0 }}
                exit={{ opacity: 0, y: -20 }}
                transition={{
                  duration: 0.4,
                  delay: index * 0.1, // appear one by one
                }}
              >
                <AwardCard
                  img={item.image_url}
                  title={item.title}
                  year={item.year}
                  awardingBody={item.description}
                />
              </motion.div>
            ))
          ) : (
            <motion.p
              key="no-awards"
              className="text-center col-span-full text-gray-500"
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
            >
              No awards found for {selectedYear}.
            </motion.p>
          )}
        </AnimatePresence>
      </div>
    </>
  );
}
