"use client";

import { useEffect, useState } from "react";
import { motion } from "framer-motion";
import ServiceCard from "@/components/ServiceCard";

interface Service {
  name: string;
  description: string;
  slug: string;
  image_url: string;
}

const BASE_URL = process.env.NEXT_PUBLIC_BASE_URL;

export default function ServicesSection() {
  const [services, setServices] = useState<Service[]>([]);
  const [activeIndex, setActiveIndex] = useState(0);

  useEffect(() => {
    async function fetchServices() {
      try {
        const res = await fetch(`${BASE_URL}/api/v1/services/active`);
        if (!res.ok) throw new Error("Failed to fetch services");
        const json = await res.json();
        setServices(json.data || []);
      } catch (err) {
        console.error("Error loading services:", err);
      }
    }
    fetchServices();
  }, []);

  return (
    <section className="pb-[80px] sm:pb-[100px] lg:pb-[130px] bg-white overflow-hidden">
      <div className="container mx-auto px-4 text-center">
        <motion.span
          className="sub-header"
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.6, ease: "easeOut" }}
        >
          Products and Solutions
        </motion.span>

        <motion.h2
          className="text-4xl sm:text-6xl lg:text-7xl xl:text-6xl font-bold mb-10 md:mb-12"
          initial={{ opacity: 0, y: 30 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.8, ease: "easeOut" }}
        >
          Driving Agri-business Excellence
        </motion.h2>

        {services.length > 0 ? (
          <motion.div
            className="flex flex-col lg:flex-row gap-3 lg:gap-3 mt-10 h-auto lg:h-[500px]"
            initial={{ opacity: 0, y: 50 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true, amount: 0.3 }}
            transition={{ duration: 0.9, ease: "easeOut" }}
          >
            {services.map((item, index) => (
              <ServiceCard
                key={index}
                img={item.image_url}
                service={item.name}
                description1={item.description}
                number={index + 1}
                link={`/services/${item.slug}`}
                active={index === activeIndex}
                onActivate={() => setActiveIndex(index)}
              />
            ))}
          </motion.div>
        ) : (
          <p className="text-gray-500 mt-10">Loading services...</p>
        )}
      </div>
    </section>
  );
}
