"use client";

import { useState, useEffect } from "react";
import Image from "next/image";

const BASE_URL = process.env.NEXT_PUBLIC_BASE_URL;

export default function InquireForm({ serviceName }: { serviceName: string }) {
  const [form, setForm] = useState({
    name: "",
    email: "",
    phone_no: "",
    message: "",
  });

  const [errors, setErrors] = useState({
    name: "",
    email: "",
    phone_no: "",
    message: "",
  });

  const [loading, setLoading] = useState(false);
  const [status, setStatus] = useState<{
    type: "success" | "error";
    message: string;
  } | null>(null);

  useEffect(() => {
    if (status) {
      const timer = setTimeout(() => {
        setStatus(null);
      }, 5000);

      return () => clearTimeout(timer);
    }
  }, [status]);

  const handleChange = (
    e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>
  ) => {
    setForm({ ...form, [e.target.id]: e.target.value });
    setErrors({ ...errors, [e.target.id]: "" }); // reset error on change
  };

  const validate = () => {
    let valid = true;
    const newErrors = { name: "", email: "", phone_no: "", message: "" };

    if (!form.name.trim()) {
      newErrors.name = "Name is required";
      valid = false;
    }

    if (!form.email.trim()) {
      newErrors.email = "Email is required";
      valid = false;
    } else if (!/\S+@\S+\.\S+/.test(form.email)) {
      newErrors.email = "Invalid email address";
      valid = false;
    }

    if (!form.phone_no.trim()) {
      newErrors.phone_no = "Phone number is required";
      valid = false;
    } else if (!/^(\+94|0)\d{9}$/.test(form.phone_no.replace(/\s+/g, ""))) {
      newErrors.phone_no =
        "Phone must start with +94 or 0 and have 9 digits after";
      valid = false;
    }

    if (!form.message.trim()) {
      newErrors.message = "Message is required";
      valid = false;
    }

    setErrors(newErrors);
    return valid;
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!validate()) return;

    setLoading(true);
    setStatus(null);

    try {
      const res = await fetch(`${BASE_URL}/api/v1/service-inquiry/submit`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          service_name: serviceName,
          name: form.name,
          email: form.email,
          phone_no: form.phone_no,
          message: form.message,
        }),
      });

      const json = await res.json();

      if (json.success) {
        setStatus({ type: "success", message: json.message });
        setForm({ name: "", email: "", phone_no: "", message: "" });
      } else {
        throw new Error(json.message || "Failed to send inquiry");
      }
    } catch (err) {
      const message =
        err instanceof Error ? err.message : "Something went wrong";
      setStatus({ type: "error", message });
    } finally {
      setLoading(false);
    }
  };

  return (
    <section className="relative pb-[80px] md:pb-[130px] bg-white">
      <div className="container mx-auto px-4 relative">
        <div className="flex flex-col lg:flex-row items-start md:items-center gap-10 md:gap-16">
          {/* Left Content */}
          <div className="relative flex-1 w-full">
            <div className="text-center lg:text-left">
              <span className="sub-header">Inquire</span>
              <h2 className="text-3xl sm:text-5xl md:text-6xl font-bold font-philosopher mb-8 md:mb-10">
                Inquire Now
              </h2>
            </div>

            {status?.message && (
              <p
                className={`mb-5 px-5 py-[15px] rounded-[15px] text-sm md:text-base ${
                  status?.type === "success"
                    ? "bg-green-100 text-green-700"
                    : "bg-red-100 text-red-700"
                }`}
              >
                {status?.message}
              </p>
            )}

            <form
              onSubmit={handleSubmit}
              className="bg-white rounded-[30px] flex flex-col gap-5"
            >
              {/* Company Name */}
              <div>
                <label
                  className="block text-sm font-medium mb-2"
                  htmlFor="name"
                >
                  Company Name
                </label>
                <input
                  id="name"
                  value={form.name}
                  onChange={handleChange}
                  type="text"
                  placeholder="Enter Company Name"
                  className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none ${
                    errors.name
                      ? "border-red-500"
                      : "border-gray-300 focus:ring-[#ac9242]"
                  }`}
                />
                {errors.name && (
                  <p
                    className="text-red-600 text-sm mt-1"
                    style={{ fontSize: "14px" }}
                  >
                    {errors.name}
                  </p>
                )}
              </div>

              {/* Email + Phone */}
              <div className="flex flex-col sm:flex-row gap-5">
                <div className="flex-1">
                  <label
                    className="block text-sm font-medium mb-2"
                    htmlFor="email"
                  >
                    Email
                  </label>
                  <input
                    id="email"
                    value={form.email}
                    onChange={handleChange}
                    type="email"
                    placeholder="Enter Company Email"
                    className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none ${
                      errors.email
                        ? "border-red-500"
                        : "border-gray-300 focus:ring-[#ac9242]"
                    }`}
                  />
                  {errors.email && (
                    <p
                      className="text-red-600 text-sm mt-1"
                      style={{ fontSize: "14px" }}
                    >
                      {errors.email}
                    </p>
                  )}
                </div>

                <div className="flex-1">
                  <label
                    className="block text-sm font-medium mb-2"
                    htmlFor="phone_no"
                  >
                    Phone Number
                  </label>
                  <input
                    id="phone_no"
                    value={form.phone_no}
                    onChange={handleChange}
                    type="tel"
                    placeholder="Enter Phone Number"
                    className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none ${
                      errors.phone_no
                        ? "border-red-500"
                        : "border-gray-300 focus:ring-[#ac9242]"
                    }`}
                  />
                  {errors.phone_no && (
                    <p
                      className="text-red-600 text-sm mt-1"
                      style={{ fontSize: "14px" }}
                    >
                      {errors.phone_no}
                    </p>
                  )}
                </div>
              </div>

              {/* Message */}
              <div>
                <label
                  className="block text-sm font-medium mb-2"
                  htmlFor="message"
                >
                  Message
                </label>
                <textarea
                  id="message"
                  value={form.message}
                  onChange={handleChange}
                  rows={5}
                  placeholder="Enter Message"
                  className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none ${
                    errors.message
                      ? "border-red-500"
                      : "border-gray-300 focus:ring-[#ac9242]"
                  }`}
                ></textarea>
                {errors.message && (
                  <p
                    className="text-red-600 text-sm mt-1"
                    style={{ fontSize: "14px" }}
                  >
                    {errors.message}
                  </p>
                )}
              </div>

              <button
                type="submit"
                className="primary-btn w-full sm:w-auto"
                disabled={loading}
              >
                {loading ? "Sending..." : "Send Message"}
              </button>
            </form>
          </div>

          {/* Right Image */}
          <div className="flex-1 w-full">
            <Image
              src="/images/about/about-02.png"
              alt="Inquiry"
              width={400}
              height={400}
              className="w-full h-auto lg:min-h-[750px] object-cover rounded-[30px]"
            />
          </div>
        </div>
      </div>
    </section>
  );
}
