"use client";

import Image from "next/image";

interface ServiceCardProps {
  img: string;
  service: string;
  description1: string;
  active: boolean;
  number?: number;
  link?: string;
  onActivate: () => void;
}

export default function ServiceCard({
  img,
  service,
  description1,
  active,
  number,
  link = "#",
  onActivate,
}: ServiceCardProps) {
  return (
    <div
      onMouseEnter={onActivate}
      onClick={onActivate} // For mobile tap activation
      className={`relative rounded-[20px] md:rounded-[30px] overflow-hidden shadow-lg cursor-pointer transition-all duration-600 ease-in-out flex-shrink-0
      w-full 
      lg:h-[500px]
      ${
        active
          ? "h-[450px] lg:h-[480px] lg:flex-[2]"
          : "h-[100px] lg:h-[220px] lg:flex-[0.5]"
      }
      `}
    >
      {/* Background Image */}
      <Image
        src={img}
        alt={service}
        width={800}
        height={600}
        className="w-full h-full object-cover transition-all duration-700"
      />

      {/* Overlay */}
      <div
        className={`absolute inset-0 transition-all duration-500 ${
          active
            ? "bg-black/60"
            : "bg-gradient-to-t from-[#0c0c0c] to-transparent"
        }`}
      />

      {/* Active Card Content */}
      <div
        className={`absolute inset-0 flex flex-col items-start justify-center text-left text-white p-[25px] lg:px-[100px] transition-all duration-600 ease-in-out ${
          active
            ? "opacity-100 translate-y-0 delay-300"
            : "opacity-0 translate-y-10 pointer-events-none"
        }`}
      >
        <h4 className="mb-4 text-3xl sm:text-4xl md:text-[40px] font-bold leading-[45px] font-philosopher">
          {service}
        </h4>
        <p className="text-sm sm:text-base mb-6 font-cabin">
          {description1.length > 300
            ? description1.substring(0, 300) + "..."
            : description1}
        </p>
        <a href={link} className="primary-btn">
          Learn More
        </a>
      </div>

      {/* Inactive Rotated Title (Desktop Only) */}
      {!active && (
        <div className="hidden lg:flex absolute bottom-0 left-0 right-0 top-30 lg:top-40 xl:top-30 w-full h-full rotate-[-90deg] items-center gap-5 text-left transition-all duration-700 ease-in-out">
          {number !== undefined && (
            <h4 className="text-white font-bold font-philosopher text-[40px]">
              {String(number).padStart(2, "0")}
            </h4>
          )}
          <h4 className="text-white max-w-[150px] whitespace-nowrap font-philosopher text-[30px] font-bold leading-[40px]">
            {service}
          </h4>
        </div>
      )}

      {/* Mobile Title (shown when inactive) */}
      {!active && (
        <div className="lg:hidden absolute bottom-6 left-6 right-6 text-white text-left transition-all duration-700">
          <h4 className="text-2xl font-bold font-philosopher">{service}</h4>
        </div>
      )}
    </div>
  );
}
