"use client";

import { Swiper, SwiperSlide } from "swiper/react";
import { Autoplay, Controller } from "swiper/modules";
import { FaArrowLeft, FaArrowRight } from "react-icons/fa6";
import Image from "next/image";
import { useState, useRef } from "react";
import { motion, useInView } from "framer-motion";
import type { Swiper as SwiperType } from "swiper";
import "swiper/css";

type Milestone = {
  year: string;
  title: string;
  description: string;
  image_url: string;
};

export default function Milestones({
  milestones,
}: {
  milestones: Milestone[];
}) {
  const [yearSwiper, setYearSwiper] = useState<SwiperType | null>(null);
  const [imageSwiper, setImageSwiper] = useState<SwiperType | null>(null);

  const handleNext = () => yearSwiper?.slideNext();
  const handlePrev = () => yearSwiper?.slidePrev();

  // 👇 Ref + inView for scroll animation trigger
  const ref = useRef(null);
  const isInView = useInView(ref, { once: true, margin: "-100px" });

  return (
    <motion.div
      ref={ref}
      initial={{ opacity: 0, y: 50 }}
      animate={isInView ? { opacity: 1, y: 0 } : {}}
      transition={{ duration: 0.9, ease: "easeOut" }}
      className="flex flex-col lg:flex-row items-center justify-center gap-10 md:gap-16 md:px-8 lg:px-12"
    >
      {/* Left (Years Section) */}
      <div className="flex-1 w-full overflow-hidden text-center lg:text-left">
        <span className="sub-header text-sm sm:text-base">Milestones</span>
        <h2 className="text-3xl sm:text-4xl md:text-6xl lg:text-7xl font-bold font-philosopher mb-5 lg:mb-2 leading-tight">
          A History of Successive Expansions
        </h2>
        <p className="text-[#909090] leading-relaxed font-cabin mb-8 sm:mb-10 lg:mb-12 text-sm sm:text-base md:text-lg">
          The roots of Pussalla date back to 1988, when it began as a small
          family-owned chicken farm. This was followed by four decades of
          successful transformation, expanding into breeding and hatchery, meat
          processing, as well as retail shops and poultry feed production, while
          upgrading into state-of-the-art automated plants.
        </p>

        <Swiper
          modules={[Autoplay, Controller]}
          onSwiper={setYearSwiper}
          controller={{ control: imageSwiper }}
          loop={true}
          slidesPerView={1}
          slidesPerGroup={1}
          allowTouchMove={false}
          breakpoints={{
            640: { slidesPerView: 2 },
            768: { slidesPerView: 2 },
            1024: { slidesPerView: 3 },
          }}
          className="w-full"
        >
          {milestones.map((timeline, i) => (
            <SwiperSlide key={i} className="flex justify-center">
              {({ isActive }) => (
                <div className="flex flex-col pl-5 pr-10 gap-2 border-l w-full">
                  <h4
                    className={`font-semibold text-[28px] sm:text-[34px] md:text-[40px] font-black ${
                      isActive ? "text-[#ac9242]" : "text-gray-700"
                    }`}
                  >
                    {timeline.year}
                  </h4>
                  <p className="text-[#909090] font-medium text-sm sm:text-base md:text-lg">
                    {timeline.title}
                  </p>
                </div>
              )}
            </SwiperSlide>
          ))}
        </Swiper>

        {/* Navigation Buttons */}
        <div className="hidden md:flex justify-center lg:justify-start gap-5 mt-8 md:mt-10">
          <button
            onClick={handlePrev}
            className="px-3 py-3 bg-[#ac9242] text-white rounded-full hover:bg-[#8a7534] transition"
          >
            <FaArrowLeft size={22} />
          </button>
          <button
            onClick={handleNext}
            className="px-3 py-3 bg-[#ac9242] text-white rounded-full hover:bg-[#8a7534] transition"
          >
            <FaArrowRight size={22} />
          </button>
        </div>
      </div>

      {/* Right (Image Section) */}
      <div className="flex-1 w-full overflow-hidden rounded-[30px]">
        <Swiper
          modules={[Autoplay, Controller]}
          onSwiper={setImageSwiper}
          controller={{ control: yearSwiper }}
          loop={true}
          slidesPerView={1}
          slidesPerGroup={1}
          allowTouchMove={false}
          className="w-full"
        >
          {milestones.map((timeline, i) => (
            <SwiperSlide key={i} className="flex justify-center">
              <motion.div
                initial={{ opacity: 0, scale: 0.95 }}
                animate={isInView ? { opacity: 1, scale: 1 } : {}}
                transition={{ duration: 0.8, ease: "easeOut", delay: 0.3 }}
                className="relative overflow-hidden shadow-lg w-full max-w-[500px] sm:max-w-[600px] md:max-w-none"
              >
                <Image
                  src={timeline.image_url}
                  alt={timeline.title}
                  className="w-full object-cover aspect-16/10 max-h-[600px] h-[300px] md:h-[550px]"
                  width={600}
                  height={600}
                />
                <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/30 to-transparent"></div>
                <div className="absolute bottom-6 sm:bottom-8 md:bottom-10 left-6 sm:left-8 md:left-10 right-6 sm:right-8 md:right-10 text-white">
                  <p className="text-sm sm:text-base md:text-lg text-center md:text-left leading-relaxed">
                    {timeline.description}
                  </p>
                </div>
              </motion.div>
            </SwiperSlide>
          ))}
        </Swiper>

        <div className="flex justify-center md:justify-start gap-5 mt-8 md:mt-10 md:hidden">
          <button
            onClick={handlePrev}
            className="px-3 py-3 bg-[#ac9242] text-white rounded-full hover:bg-[#8a7534] transition"
          >
            <FaArrowLeft size={22} />
          </button>
          <button
            onClick={handleNext}
            className="px-3 py-3 bg-[#ac9242] text-white rounded-full hover:bg-[#8a7534] transition"
          >
            <FaArrowRight size={22} />
          </button>
        </div>
      </div>
    </motion.div>
  );
}
