"use client";

import { Swiper, SwiperSlide } from "swiper/react";
import { Navigation, Pagination, Autoplay, EffectFade } from "swiper/modules";
import Image from "next/image";
import {
  FaFacebookF,
  FaInstagram,
  FaLinkedinIn,
  FaYoutube,
} from "react-icons/fa";
import { HiChevronDoubleDown } from "react-icons/hi";
import { motion } from "framer-motion";

import "swiper/css";
import "swiper/css/navigation";
import "swiper/css/pagination";
import "swiper/css/effect-fade";
import Link from "next/link";

interface Slider {
  title: string;
  description: string;
  background_image_url: string;
  image_url: string;
}

interface HeroSliderProps {
  slides: Slider[];
}

export default function HeroSlider({ slides }: HeroSliderProps) {
  if (!slides || slides.length === 0) {
    return <p className="text-center py-20">No sliders found</p>;
  }

  return (
    <motion.section
      className="relative overflow-visible"
      initial={{ opacity: 0, y: 50 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 1, ease: "easeOut" }}
    >
      <Swiper
        modules={[Navigation, Pagination, Autoplay, EffectFade]}
        navigation
        pagination
        effect={"fade"}
        fadeEffect={{ crossFade: true }}
        loop
        speed={1000}
        autoplay={{ delay: 5000, disableOnInteraction: false }}
        className="h-[600px] sm:h-[700px] lg:h-[800px] 2xl:h-[850px] rounded-b-[30px] lg:rounded-b-[100px] hero-swiper"
      >
        {slides.map((slide, index) => (
          <SwiperSlide key={index} className="relative overflow-hidden">
            {/* Background Image with Zoom */}
            <motion.div
              key={slide.background_image_url}
              className="absolute inset-0 bg-cover bg-center-top bg-no-repeat"
              style={{
                backgroundImage: `url('${slide.background_image_url}')`,
              }}
              initial={{ scale: 1 }}
              animate={{ scale: 1.05 }}
              transition={{
                duration: 15,
                repeat: Infinity,
                repeatType: "reverse",
                ease: "easeInOut",
              }}
            />

            {/* Overlay */}
            <div className="absolute inset-0 bg-black/30 rounded-b-[30px] lg:rounded-b-[100px]" />

            {/* Content */}
            <div className="relative z-10 mx-auto px-4 h-full flex items-center justify-center">
              <div className="flex flex-col lg:flex-row items-center lg:items-center gap-0 lg:gap-12 w-[100%] sm:w-[80%] md:w-[75%] lg:w-[85%] xl:w-[80%] 2xl:w-[80%] text-container">
                {/* Text Content */}
                <div className="flex-auto text-white space-y-6 text-center lg:text-left w-full lg:w-100 2xl:w-75 pt-10 lg:pt-20">
                  <h1 className="text-3xl sm:text-4xl md:text-5xl lg:text-7xl xl:text-8xl font-philosopher font-bold leading-tight mb-[10px] lg:mb-[24px]">
                    {slide.title}
                  </h1>
                  <p className="text-sm sm:text-base md:text-lg font-cabin mx-auto lg:mx-0 mb-[25px] lg:mb-12 w-full lg:w-4/5">
                    {slide.description}
                  </p>

                  <div className="flex flex-col items-center lg:items-start gap-6 lg:gap-10">
                    {/* Buttons */}
                    <div className="flex flex-row justify-center lg:justify-start gap-3 sm:gap-4 w-full">
                      <Link
                        href="/about/story"
                        className="primary-btn sm:w-auto"
                      >
                        Explore More
                      </Link>
                      <Link
                        href="/about/csr"
                        className="secondary-btn sm:w-auto"
                      >
                        Learn More
                      </Link>
                    </div>

                    {/* Social Media */}
                    <div className="lg:flex lg:gap-3 lg:text-lg hidden lg:justify-center lg:justify-start">
                      {[
                        {
                          icon: FaFacebookF,
                          href: "https://www.facebook.com/meatshop.pussalla/",
                        },
                        {
                          icon: FaLinkedinIn,
                          href: "https://lk.linkedin.com/company/pussallameatproducers",
                        },
                        {
                          icon: FaInstagram,
                          href: "https://www.instagram.com/meatshop.pussalla/?hl=en",
                        },
                        {
                          icon: FaYoutube,
                          href: "https://www.youtube.com/@pussallasausages7409",
                        },
                      ].map(({ icon: Icon, href }, i) => (
                        <a
                          key={i}
                          href={href}
                          target="_blank"
                          rel="noopener noreferrer"
                          className="w-9 h-9 sm:w-10 sm:h-10 flex items-center justify-center rounded-full bg-white/20 hover:bg-[#000] transition"
                        >
                          <Icon />
                        </a>
                      ))}
                    </div>
                  </div>
                </div>

                {/* Spacer */}
                <div className="flex-auto text-white space-y-6 text-center lg:text-left w-full lg:w-15 2xl:w-25"></div>
              </div>
            </div>

            {/* Hero Image */}
            {slide.image_url && (
              <div className="absolute z-0 lg:bottom-0 lg:right-20 lg:w-[45%] lg:max-w-xl xl:w-[50%] xl:max-w-3xl w-3/4 sm:w-2/3 bottom-0 left-1/2 transform -translate-x-1/2 lg:translate-x-0">
                <Image
                  src={slide.image_url}
                  alt={slide.title || "Slider Image"}
                  width={1200}
                  height={1200}
                  className="hidden lg:block object-contain w-full h-auto"
                  priority
                />
              </div>
            )}
          </SwiperSlide>
        ))}
      </Swiper>

      {/* Scroll Icon */}
      <div className="absolute left-1/2 -translate-x-1/2 bottom-[-30px] md:bottom-[-40px] z-50">
        <a
          href="#about"
          onClick={(e) => {
            e.preventDefault();
            const section = document.querySelector("#about");
            if (section) section.scrollIntoView({ behavior: "smooth" });
          }}
          className="bg-[#ac9242] rounded-full p-3 md:p-5 flex items-center justify-center animate-bounce"
        >
          <HiChevronDoubleDown className="text-white text-4xl md:text-4xl" />
        </a>
      </div>
    </motion.section>
  );
}
