"use client";

import { useState, useEffect } from "react";
import Image from "next/image";
import Link from "next/link";
import { motion, AnimatePresence } from "framer-motion";
import { usePathname } from "next/navigation"; // Import to get current route

interface Service {
  name: string;
  description: string;
  slug: string;
  image_url: string;
}

interface SubmenuItem {
  href: string;
  label: string;
}

export default function Navbar() {
  const pathname = usePathname(); //Current route path

  const [showHeader, setShowHeader] = useState(true);
  const [lastScrollY, setLastScrollY] = useState(0);

  const [isOpen, setIsOpen] = useState(false);
  const [aboutOpen, setAboutOpen] = useState(false);
  const [servicesOpen, setServicesOpen] = useState(false);
  const [services, setServices] = useState<Service[]>([]);
  const [loadingServices, setLoadingServices] = useState(false);

  // Close mobile menu whenever the route changes
  useEffect(() => {
    setIsOpen(false);
    setAboutOpen(false);
    setServicesOpen(false);
  }, [pathname]);

  useEffect(() => {
    const fetchServices = async () => {
      try {
        setLoadingServices(true);
        const res = await fetch(
          `${process.env.NEXT_PUBLIC_BASE_URL}/api/v1/services/active`
        );
        const data = await res.json();
        if (res.ok && data.success) {
          setServices(data.data || []);
        }
      } catch (err) {
        console.error("Failed to fetch services:", err);
      } finally {
        setLoadingServices(false);
      }
    };
    fetchServices();
  }, []);

  const navItems = [
    { href: "/", label: "Home" },
    {
      href: "#",
      label: "About Us",
      submenu: [
        { href: "/about/story", label: "Our Story" },
        // { href: "/about/commitee", label: "Our Committee" },
        { href: "/about/awards", label: "Awards and Certifications" },
        // { href: "/about/csr", label: "CSR" },
      ],
    },
    { href: "#", label: "Services", dynamicSubmenu: true },
    { href: "/careers", label: "Careers" },
    { href: "/sustainability", label: "Sustainability" },
    { href: "/blog", label: "Blogs" },
    { href: "/contact", label: "Contact Us" },
  ];

  const isActive = (
    href: string,
    submenu?: SubmenuItem[],
    dynamicSubmenu?: boolean
  ) => {
    if (href === "/") return pathname === "/";

    // If submenu exists → check if any child matches
    if (submenu) {
      return submenu.some((sub) => pathname.startsWith(sub.href));
    }

    // Dynamic submenu (services)
    if (dynamicSubmenu) {
      return pathname.startsWith("/services/");
    }

    // Normal link
    return href !== "#" && pathname.startsWith(href);
  };

  useEffect(() => {
    const handleScroll = () => {
      if (window.scrollY > lastScrollY) {
        // scrolling down → hide header
        setShowHeader(false);
      } else {
        // scrolling up → show header
        setShowHeader(true);
      }
      setLastScrollY(window.scrollY);
    };

    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, [lastScrollY]);

  return (
    <motion.header
      animate={{
        y: showHeader ? 0 : -150,
        opacity: showHeader ? 1 : 0,
      }}
      transition={{
        duration: 0.8,
        ease: "easeInOut",
      }}
      className="fixed top-0 w-full z-999"
    >
      <div className="bg-white shadow-lg rounded-[20px] lg:rounded-[30px] mt-5 mx-4 md:mx-4 lg:mx-5 xl:mx-10 2xl:mx-20 px-5 lg:px-10 py-3 lg:py-4 flex items-center justify-between">
        {/* Logo */}
        <Link href="/" className="flex-shrink-0">
          <Image
            src="/images/logo.png"
            alt="logo"
            width={140}
            height={76}
            className="transition-all duration-300 ease-out w-[80px] md:w-[140px] h-auto"
          />
        </Link>

        {/* Desktop Menu */}
        <nav className="hidden lg:flex items-center gap-10 ml-auto">
          <ul className="flex gap-8 items-center">
            {navItems.map((item, idx) => (
              <li key={idx} className="relative group">
                {item.label === "Contact Us" ? (
                  <Link
                    href={item.href}
                    className={`px-6 py-3 rounded-full font-bold border border-[#ac9242] transition-all ${
                      isActive(item.href)
                        ? "bg-transparent text-[#ac9242] hover:bg-[#ac9242] hover:text-white"
                        : "bg-[#ac9242] text-white hover:text-[#ac9242] hover:bg-transparent hover:border hover:border-[#ac9242]"
                    }`}
                  >
                    {item.label}
                  </Link>
                ) : (
                  <Link
                    href={item.href || "#"}
                    className={`font-bold text-lg transition-colors ${
                      isActive(item.href, item.submenu, item.dynamicSubmenu)
                        ? "text-[#ac9242]"
                        : "text-gray-900 hover:text-[#ac9242]"
                    }`}
                  >
                    {item.label}
                  </Link>
                )}

                {/* Static Submenu */}
                {item.submenu && (
                  <AnimatePresence>
                    <motion.ul
                      initial={{ opacity: 0, y: -10 }}
                      animate={{ opacity: 1, y: 0 }}
                      exit={{ opacity: 0, y: -10 }}
                      transition={{ duration: 0.2 }}
                      className="absolute left-0 mt-2 w-58 bg-white shadow-lg rounded-[20px] opacity-0 group-hover:opacity-100 group-hover:visible invisible transition-all p-2"
                    >
                      {item.submenu.map((sub, subIdx) => (
                        <li key={subIdx}>
                          <Link
                            href={sub.href}
                            className={`block px-4 py-2 font-medium transition-all ${
                              isActive(sub.href)
                                ? "bg-[#ac9242] text-white rounded-[10px]"
                                : "text-gray-700 hover:bg-[#ac9242] hover:text-white rounded-[10px]"
                            }`}
                          >
                            {sub.label}
                          </Link>
                        </li>
                      ))}
                    </motion.ul>
                  </AnimatePresence>
                )}

                {/* Dynamic Services Submenu */}
                {item.dynamicSubmenu && (
                  <AnimatePresence>
                    <motion.ul
                      initial={{ opacity: 0, y: -10 }}
                      animate={{ opacity: 1, y: 0 }}
                      exit={{ opacity: 0, y: -10 }}
                      transition={{ duration: 0.2 }}
                      className="absolute left-0 mt-2 w-72 bg-white shadow-lg rounded-[20px] opacity-0 group-hover:opacity-100 group-hover:visible invisible transition-all p-2"
                    >
                      {loadingServices ? (
                        <li className="px-4 py-2 text-gray-500 text-sm">
                          Loading services...
                        </li>
                      ) : services.length > 0 ? (
                        services.map((service, i) => (
                          <li key={i}>
                            <Link
                              href={`/services/${service.slug}`}
                              className={`block px-4 py-2 font-medium transition-all ${
                                pathname === `/services/${service.slug}`
                                  ? "bg-[#ac9242] text-white rounded-[10px]"
                                  : "text-gray-700 hover:bg-[#ac9242] hover:text-white rounded-[10px]"
                              }`}
                            >
                              {service.name}
                            </Link>
                          </li>
                        ))
                      ) : (
                        <li className="px-4 py-2 text-gray-500 text-sm">
                          No active services
                        </li>
                      )}
                    </motion.ul>
                  </AnimatePresence>
                )}
              </li>
            ))}
          </ul>
        </nav>

        {/* Mobile Menu Toggle */}
        <div className="lg:hidden flex items-center border border-[#101828] rounded-full p-1">
          <button
            onClick={() => setIsOpen(!isOpen)}
            className="text-gray-900 focus:outline-none"
          >
            <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
              {isOpen ? (
                <path
                  fillRule="evenodd"
                  d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z"
                  clipRule="evenodd"
                />
              ) : (
                <path
                  fillRule="evenodd"
                  d="M3 5a1 1 0 011-1h12a1 1 0 110 2H4a1 1 0 01-1-1zM3 10a1 1 0 011-1h12a1 1 0 110 2H4a1 1 0 01-1-1zM3 15a1 1 0 011-1h12a1 1 0 110 2H4a1 1 0 01-1-1z"
                  clipRule="evenodd"
                />
              )}
            </svg>
          </button>
        </div>
      </div>

      {/* Mobile Menu */}
      <AnimatePresence>
        {isOpen && (
          <motion.nav
            initial={{ opacity: 0, height: 0 }}
            animate={{ opacity: 1, height: "auto" }}
            exit={{ opacity: 0, height: 0 }}
            transition={{ duration: 0.3 }}
            className="lg:hidden bg-white shadow-lg rounded-[15px] mx-4 mt-2 overflow-hidden"
          >
            <ul className="flex flex-col gap-1 py-4 px-5">
              {navItems.map((item, idx) => (
                <li key={idx} className="relative">
                  <div className="flex justify-between items-center">
                    <Link
                      href={item.href || "#"}
                      onClick={(e) => {
                        if (item.submenu || item.dynamicSubmenu) {
                          e.preventDefault(); // prevent page navigation
                          if (item.label === "About Us")
                            setAboutOpen(!aboutOpen);
                          if (item.label === "Services")
                            setServicesOpen(!servicesOpen);
                          return;
                        }

                        setIsOpen(false); // normal links close mobile menu
                      }}
                      className={`block font-semibold py-2 ${
                        isActive(item.href, item.submenu, item.dynamicSubmenu)
                          ? "text-[#ac9242]"
                          : "text-gray-900 hover:text-[#ac9242]"
                      }`}
                    >
                      {item.label}
                    </Link>

                    {(item.submenu || item.dynamicSubmenu) && (
                      <button
                        onClick={() =>
                          item.label === "About Us"
                            ? setAboutOpen(!aboutOpen)
                            : setServicesOpen(!servicesOpen)
                        }
                        className="px-2 py-1 focus:outline-none"
                      >
                        <svg
                          className={`w-5 h-5 transform transition-transform duration-300 ${
                            (
                              item.label === "About Us"
                                ? aboutOpen
                                : servicesOpen
                            )
                              ? "rotate-180"
                              : ""
                          }`}
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M19 9l-7 7-7-7"
                          />
                        </svg>
                      </button>
                    )}
                  </div>

                  {/* Submenu: About Us */}
                  <AnimatePresence>
                    {item.submenu && aboutOpen && (
                      <motion.ul
                        initial={{ opacity: 0, height: 0 }}
                        animate={{ opacity: 1, height: "auto" }}
                        exit={{ opacity: 0, height: 0 }}
                        transition={{ duration: 0.3 }}
                        className="pl-4 mt-1 flex flex-col gap-1 overflow-hidden"
                      >
                        {item.submenu.map((sub, subIdx) => (
                          <li key={subIdx}>
                            <Link
                              href={sub.href}
                              className={`block py-2 ${
                                isActive(sub.href)
                                  ? "text-[#ac9242]"
                                  : "text-gray-700 hover:text-[#ac9242]"
                              }`}
                            >
                              {sub.label}
                            </Link>
                          </li>
                        ))}
                      </motion.ul>
                    )}
                  </AnimatePresence>

                  {/* Submenu: Services (dynamic) */}
                  <AnimatePresence>
                    {item.dynamicSubmenu && servicesOpen && (
                      <motion.ul
                        initial={{ opacity: 0, height: 0 }}
                        animate={{ opacity: 1, height: "auto" }}
                        exit={{ opacity: 0, height: 0 }}
                        transition={{ duration: 0.3 }}
                        className="pl-4 mt-1 flex flex-col gap-1 overflow-hidden"
                      >
                        {loadingServices ? (
                          <li className="py-2 text-gray-500 text-sm">
                            Loading services...
                          </li>
                        ) : services.length > 0 ? (
                          services.map((service, i) => (
                            <li key={i}>
                              <Link
                                href={`/services/${service.slug}`}
                                className={`block py-2 ${
                                  pathname === `/services/${service.slug}`
                                    ? "text-[#ac9242]"
                                    : "text-gray-700 hover:text-[#ac9242]"
                                }`}
                              >
                                {service.name}
                              </Link>
                            </li>
                          ))
                        ) : (
                          <li className="py-2 text-gray-500 text-sm">
                            No active services
                          </li>
                        )}
                      </motion.ul>
                    )}
                  </AnimatePresence>
                </li>
              ))}
            </ul>
          </motion.nav>
        )}
      </AnimatePresence>
    </motion.header>
  );
}
