"use client";

import Image from "next/image";
import Link from "next/link";

interface CSRCardProps {
  slug: string;
  img: string;
  title: string;
  description?: string;
  date?: string;
}

export default function CSRCard({
  slug,
  img,
  title,
  description,
  date,
}: CSRCardProps) {
  // Format date if provided
  let day: string | null = null;
  let month: string | null = null;
  if (date) {
    const d = new Date(date);
    day = String(d.getDate()).padStart(2, "0");
    month = d.toLocaleString("default", { month: "short" }).toUpperCase();
  }

  // Strip HTML tags from description
  const plainDescription = description
    ? description.replace(/<[^>]+>/g, "")
    : "";

  return (
    <Link
      href={`/about/csr/${slug}`}
      className="block relative rounded-[30px] overflow-hidden shadow-lg w-full group cursor-pointer"
    >
      {/* Image */}
      <Image
        src={img}
        alt={title}
        className="w-full h-[460px] object-cover transition-transform duration-1000 group-hover:scale-110"
        width={600}
        height={460}
      />

      {/* Gradient Overlay */}
      <div className="absolute inset-0 rounded-[30px] bg-gradient-to-t from-black/60 via-black/30 to-transparent transition-all duration-1000 group-hover:from-black/90 group-hover:via-black/70"></div>

      {/* Date Box */}
      {date && (
        <div className="bg-[#EEEEEE] absolute top-6 right-6 px-4 py-4 rounded-[15px] flex flex-col items-center justify-center shadow-lg">
          <h4
            className="text-3xl font-bold leading-none"
            style={{ fontSize: "50px", lineHeight: "1" }}
          >
            {day}
          </h4>
          <p
            className="text-xs uppercase leading-none"
            style={{ letterSpacing: "1px" }}
          >
            {month}
          </p>
        </div>
      )}

      {/* Content */}
      <div className="absolute bottom-0 left-0 right-0 text-center text-white px-6 pb-6">
        <div className="transition-all duration-1000 transform group-hover:-translate-y-4">
          <h4 className="text-xl font-bold">{title}</h4>

          {/* Plain Description */}
          {plainDescription && (
            <p
              className="text-sm text-gray-200 opacity-0 max-h-0 overflow-hidden
                         transition-all duration-1000 group-hover:opacity-100 group-hover:max-h-40 group-hover:mt-2"
            >
              {plainDescription.length > 50
                ? plainDescription.slice(0, 50) + "..."
                : plainDescription}
            </p>
          )}
        </div>
      </div>
    </Link>
  );
}
