"use client";

import Image from "next/image";
import Link from "next/link";

interface BlogCardProps {
  img: string;
  date: string;
  title: string;
  description: string;
  href?: string; // dynamic link to blog detail
}

export default function BlogCard({
  img,
  date,
  title,
  description,
  href = "#",
}: BlogCardProps) {
  // Format date
  const d = new Date(date);
  const day = String(d.getDate()).padStart(2, "0");
  const month = d.toLocaleString("default", { month: "short" }).toUpperCase();

  return (
    <Link href={href} className="group">
      <div
        className="bg-[#EEEEEE] rounded-[30px] overflow-hidden flex flex-col h-full text-left 
                   transform transition-transform duration-500 ease-in-out"
      >
        {/* Image with Date Overlay */}
        <div className="relative overflow-hidden rounded-t-[30px]">
          <Image
            src={img}
            alt={title}
            width={600}
            height={400}
            className="w-full h-75 object-cover transition-transform duration-500 ease-in-out group-hover:scale-110"
          />
          <div className="bg-[#EEEEEE] absolute top-5 right-5 px-4 py-4 rounded-[15px] flex flex-col items-center justify-center shadow-lg">
            <h4
              className="text-3xl font-bold leading-none"
              style={{ fontSize: "50px", lineHeight: "1" }}
            >
              {day}
            </h4>
            <p
              className="text-xs uppercase leading-none"
              style={{ letterSpacing: "1px" }}
            >
              {month}
            </p>
          </div>
        </div>

        {/* Blog Content */}
        <div className="p-7 flex flex-col flex-grow transition-all duration-500 group-hover:-translate-y-1">
          <h4 className="text-xl font-bold mb-2 line-clamp-2 transition-colors duration-300">
            {title.length > 30 ? title.slice(0, 30) + "..." : title}
          </h4>
          <p className="text-[#909090] mb-8 md:mb-8 text-sm flex-grow">
            {description.length > 150
              ? description.slice(0, 150) + "..."
              : description}
          </p>

          {/* Read More Button */}
          <span className="primary-btn inline-block self-start cursor-pointer">
            Read More
          </span>
        </div>
      </div>
    </Link>
  );
}
