"use client";

import Link from "next/link";
import { motion, useInView } from "framer-motion";
import { useRef } from "react";

interface BannerProps {
  title: string;
  breadcrumbs: { label: string; href?: string }[];
}

export default function Banner({ title, breadcrumbs }: BannerProps) {
  const ref = useRef(null);
  const isInView = useInView(ref, { once: true, margin: "-50px" });

  return (
    <section className="relative flex items-end justify-center text-center bg-[#AD9343] bg-center rounded-b-[50px] md:rounded-b-[100px]">
      <div
        ref={ref}
        className="relative z-10 container mx-auto px-4 md:px-6 pt-35 md:pt-55 pb-15 md:pb-25 text-white"
      >
        {/* Title Animation */}
        <motion.h1
          initial={{ opacity: 0, y: 40 }}
          animate={isInView ? { opacity: 1, y: 0 } : {}}
          transition={{ duration: 0.8, ease: "easeOut" }}
          className="sm:text-5xl font-bold mb-0 md:mb-4"
        >
          {title}
        </motion.h1>

        {/* Breadcrumb Animation */}
        <motion.nav
          initial={{ opacity: 0, y: 30 }}
          animate={isInView ? { opacity: 1, y: 0 } : {}}
          transition={{ duration: 0.8, ease: "easeOut", delay: 0.3 }}
          className="text-sm sm:text-base mt-2"
        >
          <ol className="flex flex-wrap justify-center items-center gap-2 text-center">
            {breadcrumbs.map((crumb, index) => {
              const isLong = crumb.label.length > 30;

              return (
                <li
                  key={index}
                  className={`flex items-center gap-1 ${
                    isLong ? "max-w-[300px] break-words sm:max-w-none" : ""
                  }`}
                >
                  {crumb.href ? (
                    <Link
                      href={crumb.href}
                      className="hover:underline hover:text-[#ac9242]"
                    >
                      <span
                        className={`text-white ${
                          isLong ? "block sm:inline break-words" : ""
                        }`}
                      >
                        {crumb.label}
                      </span>
                    </Link>
                  ) : (
                    <span
                      className={`text-white ${
                        isLong ? "block sm:inline break-words" : ""
                      }`}
                    >
                      {crumb.label}
                    </span>
                  )}

                  {index < breadcrumbs.length - 1 && (
                    <svg
                      xmlns="http://www.w3.org/2000/svg"
                      className="w-3 h-3 text-white shrink-0"
                      fill="none"
                      viewBox="0 0 6 10"
                    >
                      <path
                        stroke="currentColor"
                        strokeLinecap="round"
                        strokeLinejoin="round"
                        strokeWidth={2}
                        d="M1 9l4-4-4-4"
                      />
                    </svg>
                  )}
                </li>
              );
            })}
          </ol>
        </motion.nav>
      </div>
    </section>
  );
}
