"use client";

import Image from "next/image";
import Link from "next/link";
import AwardSlider from "@/components/AwardSlider";
import { motion } from "framer-motion";
import { Swiper, SwiperSlide } from "swiper/react";
import { Autoplay } from "swiper/modules";
import "swiper/css";

interface Award {
  title: string;
  description: string;
  year: string;
  image_url: string;
}

interface AwardsSectionProps {
  awards: Award[];
  baseUrl: string;
}

export default function AwardsSection({ awards, baseUrl }: AwardsSectionProps) {
  if (!awards || awards.length === 0) return null;

  // HARD-CODED LEFT SLIDER DATA
  const leftSliderImages = [
    {
      image: "/images/timeline/timeline-02.png",
      title: "Most Trusted Brand 2023",
    },
    {
      image: "/images/timeline/timeline-02.png",
      title: "National Quality Award 2022",
    },
    {
      image: "/images/timeline/timeline-02.png",
      title: "Excellence in Innovation",
    },
  ];

  return (
    <section className="pt-[40px] pb-[80px] sm:py-[100px] lg:py-[130px] rounded-t-[50px] lg:rounded-t-[100px] bg-gradient-to-t from-[#FFFFFF] to-[#EEEEEE] overflow-hidden">
      <div className="container mx-auto px-4">
        <div className="flex flex-col lg:flex-row items-center gap-10 lg:gap-16">
          {/* LEFT IMAGE SLIDER */}
          <motion.div
            className="relative flex-[1] min-w-0 flex flex-col bg-[#EEEEEE] rounded-[30px] w-full h-full overflow-hidden"
            initial={{ opacity: 0, y: 80 }}
            whileInView={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, ease: "easeOut" }}
            viewport={{ once: true, amount: 0.3 }}
          >
            <Swiper
              modules={[Autoplay]}
              loop
              slidesPerView={1}
              className="w-full"
              spaceBetween={0}
              autoplay={{ delay: 4000, disableOnInteraction: false }}
            >
              {leftSliderImages.map((item, idx) => (
                <SwiperSlide key={idx}>
                  <div className="flex flex-col items-center">
                    <Image
                      src={item.image}
                      alt={item.title}
                      width={250}
                      height={400}
                      className="w-full object-cover aspect-16/10 max-h-[600px] h-[300px] md:h-[550px]"
                    />
                    <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/30 to-transparent"></div>
                    <div className="absolute bottom-6 sm:bottom-8 md:bottom-10 left-6 sm:left-8 md:left-10 right-6 sm:right-8 md:right-10 text-white">
                      <h4 className="text-sm sm:text-base md:text-lg text-center leading-relaxed">
                        {item.title}
                      </h4>
                    </div>
                  </div>
                </SwiperSlide>
              ))}
            </Swiper>
          </motion.div>

          {/* RIGHT CONTENT (CMS DATA) */}
          <motion.div
            className="flex-[1.5] min-w-0 w-full mt-0 lg:mt-0 overflow-hidden text-center lg:text-left"
            initial={{ opacity: 0, x: 100 }}
            whileInView={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.8, ease: "easeOut", delay: 0.2 }}
            viewport={{ once: true, amount: 0.3 }}
          >
            <div className="flex items-center justify-between mb-5">
              <div>
                <span className="sub-header">Awards</span>
                <h2 className="text-4xl sm:text-5xl md:text-6xl lg:text-7xl font-bold font-philosopher">
                  Recognised Excellence and Trust
                </h2>
              </div>
            </div>

            <p className="text-[#909090] leading-relaxed font-cabin mb-10 sm:mb-12 md:mb-8 text-sm sm:text-base md:text-lg">
              Pussalla’s commitment has been recognised through awards and
              honours in both national and international scale, signalling of
              promises that delivered beyond words.
            </p>

            <Link
              href="/about/awards"
              className="primary-btn whitespace-nowrap hidden lg:inline-block mb-10"
            >
              Explore More
            </Link>

            {/* CMS slider except the first image */}
            <AwardSlider awards={awards.slice(1)} baseUrl={baseUrl} />

            <Link
              href="/about/awards"
              className="primary-btn whitespace-nowrap inline-block lg:hidden mt-10"
            >
              Explore More
            </Link>
          </motion.div>
        </div>
      </div>
    </section>
  );
}
