"use client";

import { useState, useEffect } from "react";

interface ApplyFormProps {
  jobTitle: string;
}

export default function ApplyForm({ jobTitle }: ApplyFormProps) {
  const [loading, setLoading] = useState(false);
  const [message, setMessage] = useState<string | null>(null);
  const [errors, setErrors] = useState<Record<string, string>>({});

  // Auto-hide message after 5 seconds
  useEffect(() => {
    if (message) {
      const timer = setTimeout(() => {
        setMessage(null);
      }, 5000);
      return () => clearTimeout(timer);
    }
  }, [message]);

  const validateForm = (formData: FormData) => {
    const newErrors: Record<string, string> = {};

    const name = formData.get("full_name")?.toString().trim() || "";
    const email = formData.get("email")?.toString().trim() || "";
    const phone = formData.get("phone_no")?.toString().trim() || "";
    const coverLetter = formData.get("cover_letter")?.toString().trim() || "";
    const resume = formData.get("resume") as File | null;

    if (name.length < 3) {
      newErrors.full_name = "Full name must be at least 3 characters.";
    }

    if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
      newErrors.email = "Please enter a valid email address.";
    }

    if (!/^(\+94|0)\d{9}$/.test(phone)) {
      newErrors.phone_no =
        "Phone must start with +94 or 0 and have 9 digits after.";
    }

    if (coverLetter.length < 20) {
      newErrors.cover_letter = "Cover letter must be at least 20 characters.";
    }

    if (!resume) {
      newErrors.resume = "Please upload your resume.";
    } else {
      const allowedTypes = [
        "application/pdf",
        "application/msword",
        "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
      ];
      if (!allowedTypes.includes(resume.type)) {
        newErrors.resume =
          "Only PDF, DOC, or DOCX files are allowed for the resume.";
      }
      if (resume.size > 5 * 1024 * 1024) {
        newErrors.resume = "Resume file must be less than 5 MB.";
      }
    }

    return newErrors;
  };

  const handleChange = (
    e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>
  ) => {
    const { name, value } = e.target;
    const files = (e.target as HTMLInputElement).files;
    setErrors((prev) => {
      const updated = { ...prev };
      // Remove error when corrected
      if (name === "full_name" && value.trim().length >= 3)
        delete updated.full_name;
      if (name === "email" && /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value.trim()))
        delete updated.email;
      if (name === "phone_no" && /^(\+94|0)\d{9}$/.test(value.trim()))
        delete updated.phone_no;
      if (name === "cover_letter" && value.trim().length >= 20)
        delete updated.cover_letter;
      if (name === "resume" && files && files.length > 0) delete updated.resume;
      return updated;
    });
  };

  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    setMessage(null);
    setErrors({});

    const formData = new FormData(e.currentTarget);
    formData.append("job_title", jobTitle);

    const validationErrors = validateForm(formData);
    if (Object.keys(validationErrors).length > 0) {
      setErrors(validationErrors);
      return;
    }

    setLoading(true);
    try {
      const res = await fetch(
        `${process.env.NEXT_PUBLIC_BASE_URL}/api/v1/job-applicant/submit`,
        {
          method: "POST",
          body: formData,
        }
      );

      const data = await res.json();

      if (res.ok && data.success) {
        setMessage(data.message || "Application submitted successfully!");
        e.currentTarget.reset();
      } else {
        setMessage(
          data.message || "Failed to submit application. Please try again."
        );
      }
    } catch {
      setMessage("Network error. Please try again later.");
    } finally {
      setLoading(false);
    }
  };

  return (
    <form
      className="bg-white rounded-[30px] flex flex-col gap-5"
      onSubmit={handleSubmit}
      noValidate
    >
      {message && (
        <p
          className={`mb-5 px-5 py-[15px] rounded-[15px] text-sm md:text-base transition-opacity duration-500 ${
            message.includes("success")
              ? "bg-green-100 text-green-700"
              : "bg-red-100 text-red-700"
          }`}
        >
          {message}
        </p>
      )}

      {/* Full Name */}
      <div>
        <label className="block text-sm font-medium mb-2" htmlFor="name">
          Full Name
        </label>
        <input
          type="text"
          name="full_name"
          id="name"
          onChange={handleChange}
          required
          className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none focus:ring-1 ${
            errors.full_name
              ? "border-red-500 focus:ring-red-500"
              : "border-gray-300 focus:ring-[#ac9242]"
          }`}
          placeholder="Enter Your Full Name"
        />
        {errors.full_name && (
          <p className="text-red-500 text-sm mt-1" style={{ fontSize: "14px" }}>
            {errors.full_name}
          </p>
        )}
      </div>

      {/* Email & Phone */}
      <div className="flex gap-5 flex-col md:flex-row">
        <div className="flex-1">
          <label className="block text-sm font-medium mb-2" htmlFor="email">
            Email
          </label>
          <input
            type="email"
            name="email"
            id="email"
            onChange={handleChange}
            required
            className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none focus:ring-1 ${
              errors.email
                ? "border-red-500 focus:ring-red-500"
                : "border-gray-300 focus:ring-[#ac9242]"
            }`}
            placeholder="Enter Email"
          />
          {errors.email && (
            <p
              className="text-red-500 text-sm mt-1"
              style={{ fontSize: "14px" }}
            >
              {errors.email}
            </p>
          )}
        </div>

        <div className="flex-1">
          <label className="block text-sm font-medium mb-2" htmlFor="phone">
            Phone Number
          </label>
          <input
            type="tel"
            name="phone_no"
            id="phone"
            onChange={handleChange}
            required
            className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none focus:ring-1 ${
              errors.phone_no
                ? "border-red-500 focus:ring-red-500"
                : "border-gray-300 focus:ring-[#ac9242]"
            }`}
            placeholder="Enter Phone Number"
          />
          {errors.phone_no && (
            <p
              className="text-red-500 text-sm mt-1"
              style={{ fontSize: "14px" }}
            >
              {errors.phone_no}
            </p>
          )}
        </div>
      </div>

      {/* Resume */}
      <div>
        <label className="block text-sm font-medium mb-2">Upload Resume</label>
        <input
          type="file"
          name="resume"
          accept=".pdf,.doc,.docx"
          onChange={handleChange}
          required
          className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none focus:ring-1 ${
            errors.resume
              ? "border-red-500 focus:ring-red-500"
              : "border-gray-300 focus:ring-[#ac9242]"
          }`}
        />
        {errors.resume && (
          <p className="text-red-500 text-sm mt-1" style={{ fontSize: "14px" }}>
            {errors.resume}
          </p>
        )}
      </div>

      {/* Cover Letter */}
      <div>
        <label className="block text-sm font-medium mb-2" htmlFor="message">
          Cover Letter
        </label>
        <textarea
          name="cover_letter"
          id="message"
          rows={5}
          onChange={handleChange}
          required
          className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none focus:ring-1 ${
            errors.cover_letter
              ? "border-red-500 focus:ring-red-500"
              : "border-gray-300 focus:ring-[#ac9242]"
          }`}
          placeholder="Enter Your Cover Letter"
        ></textarea>
        {errors.cover_letter && (
          <p className="text-red-500 text-sm mt-1" style={{ fontSize: "14px" }}>
            {errors.cover_letter}
          </p>
        )}
      </div>

      {/* Submit */}
      <button
        type="submit"
        disabled={loading}
        className={`primary-btn ${loading ? "opacity-70" : ""}`}
      >
        {loading ? "Submitting..." : "Send Message"}
      </button>
    </form>
  );
}
