"use client";

import { useState, useEffect } from "react";
import Image from "next/image";
import Link from "next/link";
import Banner from "@/components/Banner";
import { HiPhone, HiMail, HiLocationMarker } from "react-icons/hi";
import { motion } from "framer-motion";

export default function ContactPage() {
  const BASE_URL = process.env.NEXT_PUBLIC_BASE_URL;
  const [formData, setFormData] = useState({
    inquiry_type: "inquiry",
    full_name: "",
    email: "",
    phone_no: "",
    message: "",
  });
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [status, setStatus] = useState<{
    type: "success" | "error";
    message: string;
  } | null>(null);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    if (status) {
      const timer = setTimeout(() => setStatus(null), 5000);
      return () => clearTimeout(timer);
    }
  }, [status]);

  const validate = () => {
    const newErrors: Record<string, string> = {};
    if (!formData.full_name.trim())
      newErrors.full_name = "Full name is required.";
    if (!formData.email.trim()) newErrors.email = "Email is required.";
    else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email))
      newErrors.email = "Please enter a valid email.";
    if (!formData.phone_no.trim())
      newErrors.phone_no = "Phone number is required.";
    else if (!/^(\+94\d{9}|0\d{9})$/.test(formData.phone_no))
      newErrors.phone_no =
        "Phone must start with +94 or 0 followed by 9 digits.";
    if (!formData.message.trim()) newErrors.message = "Message is required.";
    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleChange = (
    e: React.ChangeEvent<
      HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement
    >
  ) => {
    const { name, value } = e.target;
    setFormData({ ...formData, [name]: value });

    setErrors((prev) => {
      const newErrors = { ...prev };
      if (name === "full_name" && value.trim() !== "")
        delete newErrors.full_name;
      if (name === "email" && /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value))
        delete newErrors.email;
      if (name === "phone_no" && /^(\+94\d{9}|0\d{9})$/.test(value))
        delete newErrors.phone_no;
      if (name === "message" && value.trim() !== "") delete newErrors.message;
      return newErrors;
    });
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setStatus(null);
    if (!validate()) return;

    try {
      setLoading(true);
      const res = await fetch(`${BASE_URL}/api/v1/inquiry/submit`, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(formData),
      });
      const data = await res.json();
      if (data.success && data.code === 200) {
        setStatus({ type: "success", message: data.message });
        setFormData({
          inquiry_type: "inquiry",
          full_name: "",
          email: "",
          phone_no: "",
          message: "",
        });
      } else {
        setStatus({
          type: "error",
          message: data.message || "Failed to submit inquiry.",
        });
      }
    } catch (error) {
      console.error(error);
      setStatus({
        type: "error",
        message: "Something went wrong. Please try again later.",
      });
    } finally {
      setLoading(false);
    }
  };

  const fadeUp = {
    hidden: { opacity: 0, y: 30 },
    visible: { opacity: 1, y: 0, transition: { duration: 0.8 } },
  };

  // const cardVariants = {
  //   hidden: { opacity: 0, y: 20 },
  //   visible: { opacity: 1, y: 0, transition: { duration: 0.8 } },
  // };

  return (
    <>
      <Banner
        title="Contact Us"
        breadcrumbs={[{ label: "Home", href: "/" }, { label: "Contact" }]}
      />

      {/* Contact Info */}
      <motion.section
        className="py-[80px] sm:py-[100px] lg:py-[130px] bg-white"
        initial="hidden"
        whileInView="visible"
        viewport={{ once: true, amount: 0.3 }}
        variants={fadeUp}
      >
        <div className="container mx-auto px-4">
          <div className="flex flex-col gap-6 md:flex-row md:gap-10">
            {[
              {
                icon: HiPhone,
                title: "Call Us",
                content_1: "036 2255 062",
                content_2: "036 2255 062",
                link_1: "tel:0362255062",
                link_2: "tel:0362255062",
              },
              {
                icon: HiMail,
                title: "Email Us",
                content_1: "info@pussalla.com",
                content_2: "info@pussalla.com",
                link_1: "mailto:info@pussalla.com",
                link_2: "mailto:info@pussalla.com",
              },
              {
                icon: HiLocationMarker,
                title: "Visit Us",
                content_1:
                  "585, School Lane, Pelawatta,\nBattaramulla, Sri Lanka",
                link_1: "https://maps.app.goo.gl/yZiC1RYdjvwr42U59",
                link_2: "https://maps.app.goo.gl/yZiC1RYdjvwr42U59",
              },
            ].map((card, idx) => (
              <motion.div
                key={idx}
                variants={{
                  hidden: { opacity: 0, y: 30 },
                  visible: {
                    opacity: 1,
                    y: 0,
                    transition: { duration: 0.8, delay: idx * 0.2 },
                  },
                }}
                className="flex-1 bg-linear-to-t from-[#EEEEEE] to-[#F9F9F9] p-6 rounded-[20px] md:rounded-[30px] text-center flex flex-col mt-10 hover:shadow-lg transition-transform duration-300 hover:-translate-y-2"
              >
                <div className="flex justify-center -mt-13 md:-mt-15 mb-4">
                  <div className="bg-black text-white p-5 md:p-6 rounded-full">
                    <card.icon className="w-8 h-8 md:w-10 md:h-10" />
                  </div>
                </div>
                <h4 className="text-base md:text-lg font-semibold">
                  {card.title}
                </h4>
                <Link
                  href={card.link_1}
                  target={card.link_1.startsWith("http") ? "_blank" : "_self"}
                  rel="noopener noreferrer"
                >
                  <p className="text-[#909090] text-sm mt-2 whitespace-pre-line">
                    {card.content_1}
                  </p>
                </Link>
                <Link
                  href={card.link_2}
                  target={card.link_2.startsWith("http") ? "_blank" : "_self"}
                  rel="noopener noreferrer"
                >
                  <p className="text-[#909090] text-sm mt-2 whitespace-pre-line">
                    {card.content_2}
                  </p>
                </Link>
              </motion.div>
            ))}
          </div>
        </div>
      </motion.section>

      {/* Contact Form */}
      <motion.section
        className="pb-[80px] sm:pb-[100px] lg:pb-[130px] bg-white"
        initial="hidden"
        whileInView="visible"
        viewport={{ once: true, amount: 0.3 }}
      >
        <div className="container px-4 mx-auto">
          <div className="flex flex-col lg:flex-row items-start md:items-center gap-10 md:gap-16">
            {/* Left Content */}
            <motion.div className="flex-1 w-full" variants={fadeUp}>
              <div className="text-center lg:text-left">
                <span className="sub-header">Contact Us</span>
                <h2 className="text-3xl sm:text-4xl md:text-6xl xl:text-6xl font-bold font-philosopher mb-6 md:mb-10">
                  Get in touch
                </h2>
              </div>

              {status && (
                <motion.p
                  className={`mb-5 px-5 py-4 rounded-[15px] text-sm ${
                    status.type === "success"
                      ? "bg-green-100 text-green-700"
                      : "bg-red-100 text-red-700"
                  }`}
                  initial={{ opacity: 0 }}
                  animate={{ opacity: 1 }}
                >
                  {status.message}
                </motion.p>
              )}

              <motion.form
                onSubmit={handleSubmit}
                className="bg-white rounded-[20px] md:rounded-[30px] flex flex-col gap-5"
                variants={fadeUp}
              >
                <div className="flex flex-col gap-1">
                  <label className="text-sm font-light">Inquiry Type</label>
                  <select
                    name="inquiry_type"
                    value={formData.inquiry_type}
                    onChange={handleChange}
                    className="border border-gray-300 rounded-[15px] px-5 py-[15px] focus:outline-none focus:ring-1 focus:ring-[#ac9242]"
                  >
                    <option value="inquiry1">Inquiry Type 01</option>
                    <option value="inquiry2">Inquiry Type 02</option>
                    <option value="inquiry3">Inquiry Type 03</option>
                  </select>
                </div>

                <div className="flex flex-col gap-1">
                  <label className="text-sm font-light">Full Name</label>
                  <input
                    type="text"
                    name="full_name"
                    value={formData.full_name}
                    onChange={handleChange}
                    placeholder="Your Full Name"
                    className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none focus:ring-1 focus:ring-[#ac9242] ${
                      errors.full_name ? "border-red-500" : "border-gray-300"
                    }`}
                  />
                  {errors.full_name && (
                    <p
                      className="text-red-500 text-sm transition-opacity"
                      style={{ fontSize: "14px" }}
                    >
                      {errors.full_name}
                    </p>
                  )}
                </div>

                <div className="flex flex-col md:flex-row gap-5">
                  <div className="flex-1">
                    <label className="text-sm font-light">Phone Number</label>
                    <input
                      type="text"
                      name="phone_no"
                      value={formData.phone_no}
                      onChange={handleChange}
                      placeholder="Your Number"
                      className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none focus:ring-1 focus:ring-[#ac9242] ${
                        errors.phone_no ? "border-red-500" : "border-gray-300"
                      }`}
                    />
                    {errors.phone_no && (
                      <p
                        className="text-red-500 text-sm transition-opacity"
                        style={{ fontSize: "14px" }}
                      >
                        {errors.phone_no}
                      </p>
                    )}
                  </div>

                  <div className="flex-1">
                    <label className="text-sm font-light">Email</label>
                    <input
                      type="email"
                      name="email"
                      value={formData.email}
                      onChange={handleChange}
                      placeholder="Your Email"
                      className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none focus:ring-1 focus:ring-[#ac9242] ${
                        errors.email ? "border-red-500" : "border-gray-300"
                      }`}
                    />
                    {errors.email && (
                      <p
                        className="text-red-500 text-sm transition-opacity"
                        style={{ fontSize: "14px" }}
                      >
                        {errors.email}
                      </p>
                    )}
                  </div>
                </div>

                <div className="flex flex-col gap-1">
                  <label className="text-sm font-light">Message</label>
                  <textarea
                    name="message"
                    value={formData.message}
                    onChange={handleChange}
                    rows={5}
                    placeholder="Message"
                    className={`w-full border rounded-[15px] px-5 py-[15px] focus:outline-none focus:ring-1 focus:ring-[#ac9242] ${
                      errors.message ? "border-red-500" : "border-gray-300"
                    }`}
                  ></textarea>
                  {errors.message && (
                    <p
                      className="text-red-500 text-sm transition-opacity"
                      style={{ fontSize: "14px" }}
                    >
                      {errors.message}
                    </p>
                  )}
                </div>

                <button
                  type="submit"
                  disabled={loading}
                  className="primary-btn disabled:opacity-50"
                >
                  {loading ? "Sending..." : "Send Message"}
                </button>
              </motion.form>
            </motion.div>

            {/* Right Image */}
            <motion.div className="flex-1 w-full" variants={fadeUp}>
              <Image
                src="/images/about/about-02.png"
                alt="Contact"
                width={600}
                height={400}
                className="rounded-[20px] md:rounded-[30px] object-cover w-full h-auto lg:min-h-[800px]"
              />
            </motion.div>
          </div>
        </div>
      </motion.section>

      {/* Map */}
      <motion.section
        className="pb-[80px] sm:pb-[100px] lg:pb-[130px] bg-white"
        initial="hidden"
        whileInView="visible"
        viewport={{ once: true, amount: 0.3 }}
        variants={fadeUp}
      >
        <div className="container px-4 mx-auto">
          <div className="w-full h-[300px] lg:h-[600px] overflow-hidden rounded-[20px] md:rounded-[30px] border border-[#ac9242]">
            <iframe
              src="https://www.google.com/maps/embed?pb=!1m18!1m12!1m3!1d3960.6168230246803!2d80.14070889999999!3d6.936316!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m3!1m2!1s0x3ae3abe9a049e479%3A0x9ee45dc0e46bae08!2sPussalla%20Head%20Office!5e0!3m2!1sen!2slk!4v1758012285885!5m2!1sen!2slk"
              width="100%"
              height="100%"
              style={{ border: 0 }}
              allowFullScreen
              loading="lazy"
              referrerPolicy="no-referrer-when-downgrade"
            ></iframe>
          </div>
        </div>
      </motion.section>
    </>
  );
}
