import Banner from "@/components/Banner";
import BlogCard from "@/components/BlogCard";

export const dynamic = "force-dynamic"; // <- ADD THIS HERE

interface Blog {
  title: string;
  description: string;
  slug: string;
  created_at: string;
  featured_image_url: string;
}

interface BlogApiResponse {
  success: boolean;
  message: string;
  data: Blog[];
}

function stripHtml(html: string) {
  if (!html) return "";
  return html.replace(/<[^>]*>?/gm, "");
}

export default async function BlogPage() {
  const BASE_URL = process.env.NEXT_PUBLIC_BASE_URL;
  let blogs: Blog[] = [];

  try {
    if (!BASE_URL) throw new Error("NEXT_PUBLIC_BASE_URL is not defined");

    const res = await fetch(`${BASE_URL}/api/v1/blogs/active`, {
      cache: "no-store", // always fetch fresh data
    });

    if (!res.ok) throw new Error("Failed to fetch active blogs");

    const json: BlogApiResponse = await res.json();

    if (Array.isArray(json.data)) {
      blogs = json.data;
    }
  } catch (error) {
    console.error("Blog fetch error:", error);
  }

  // Pagination logic (can still be client-side)
  const blogsPerPage = 6;
  // const totalPages = Math.ceil(blogs.length / blogsPerPage);

  return (
    <>
      <Banner
        title="Blogs"
        breadcrumbs={[{ label: "Home", href: "/" }, { label: "Blogs" }]}
      />

      <section className="py-[80px] sm:py-[100px] lg:py-[130px] bg-white">
        <div className="container mx-auto px-4">
          {blogs.length > 0 ? (
            <>
              <div className="grid gap-6 sm:grid-cols-1 md:grid-cols-2 lg:grid-cols-3">
                {blogs.slice(0, blogsPerPage).map((blog, index) => (
                  <BlogCard
                    key={index}
                    img={blog.featured_image_url}
                    date={blog.created_at}
                    title={blog.title}
                    description={stripHtml(blog.description)}
                    href={`/blog/${blog.slug}`}
                  />
                ))}
              </div>

              {/* Pagination logic can be implemented here client-side if needed */}
            </>
          ) : (
            <p className="text-center text-gray-500">No blogs available.</p>
          )}
        </div>
      </section>
    </>
  );
}
