import Banner from "@/components/Banner";
import CSRFilter from "@/components/CSRFilter";

export const dynamic = "force-dynamic"; // <- ADD THIS HERE

interface CSRProject {
  title: string;
  category: string;
  description: string;
  slug: string;
  created_at: string;
  featured_image_url: string;
}

interface CSRApiResponse {
  success: boolean;
  message: string;
  data: Record<string, CSRProject[]> | CSRProject[] | null; // handle array or object
}

export default async function CSRPage() {
  const BASE_URL = process.env.NEXT_PUBLIC_BASE_URL;
  let csrList: CSRProject[] = [];
  let categories: string[] = [];

  try {
    if (!BASE_URL) throw new Error("NEXT_PUBLIC_BASE_URL is not defined");

    const res = await fetch(`${BASE_URL}/api/v1/csr/active`, {
      cache: "no-store",
    });

    if (!res.ok) throw new Error("Failed to fetch CSR projects");

    const json: CSRApiResponse = await res.json();

    // If data is an array (sometimes API may return a plain array)
    if (Array.isArray(json.data)) {
      csrList = json.data;
      categories = Array.from(
        new Set(csrList.map((p) => p.category || "General"))
      );
    }
    // If data is an object with categories
    else if (
      json.data &&
      typeof json.data === "object" &&
      !Array.isArray(json.data)
    ) {
      const dataObj = json.data as Record<string, CSRProject[]>;
      categories = Object.keys(dataObj);
      csrList = categories.flatMap((cat) => {
        const projects = dataObj[cat];
        if (Array.isArray(projects)) {
          return projects.map((p) => ({ ...p, category: cat }));
        }
        return [];
      });
    }
  } catch (error) {
    console.error("CSR fetch error:", error);
  }

  return (
    <>
      <Banner
        title="CSR Projects"
        breadcrumbs={[
          { label: "Home", href: "/" },
          { label: "About", href: "/about" },
          { label: "CSR" },
        ]}
      />

      <section className="py-[80px] sm:py-[100px] lg:py-[130px] bg-white">
        <div className="container mx-auto px-4">
          <span className="sub-header block text-center">CSR Projects</span>
          <h2 className="text-4xl text-center sm:text-6xl lg:text-7xl xl:text-6xl font-bold mb-5">
            Responsibility beyond Profits
          </h2>
          <p className="text-[#909090] text-center leading-relaxed font-cabin text-sm sm:text-base md:text-lg max-w-3xl mx-auto px-2 sm:px-6 md:px-10 mb-15">
            Our purpose extends beyond traditional business goals, and
            emphasises a holistic stakeholder view. Throughout our journey, we
            have made continued efforts to make our farming, sourcing, and
            energy consumption sustainable while improving the well-being of our
            own employees and external communities through numerous CSR
            initiatives. Every step we take in our operations is aims to benefit
            our employees, suppliers, animals, as well as our broader community
            and the ecosystem, while driving growth responsibly.
          </p>
          {csrList.length > 0 ? (
            <CSRFilter csrList={csrList} categories={categories} />
          ) : (
            <p className="text-center text-gray-500">
              No CSR projects available at the moment.
            </p>
          )}
        </div>
      </section>
    </>
  );
}
