import Image from "next/image";
import Banner from "@/components/Banner";
import CommiteeCard from "@/components/CommiteeCard";

export const dynamic = "force-dynamic"; // <- ADD THIS HERE

interface Leader {
  full_name: string;
  designation: string;
  message: string | null;
  is_active_message: number;
  image_url: string;
}
const BASE_URL = process.env.NEXT_PUBLIC_BASE_URL;

// Add Metadata here
export const metadata = {
  title: "Executive Committee | Pussalla",
  description:
    "Meet the leadership and executive committee behind our success — dedicated professionals guiding our organization toward excellence.",
  openGraph: {
    title: "Executive Committee | Pussalla",
    description:
      "Discover our leadership team and their vision for innovation and growth.",
    url: "https://pussalla.com/about/commitee",
    siteName: "Pussalla",
    images: [
      {
        url: "https://pussalla.com/og-image.jpg",
        width: 1200,
        height: 630,
        alt: "Executive Committee",
      },
    ],
    locale: "en_US",
    type: "website",
  },
};

async function getLeaders(): Promise<Leader[]> {
  try {
    const res = await fetch(`${BASE_URL}/api/v1/leadership-team/active`, {
      cache: "no-store",
    });

    if (!res.ok) {
      console.error("Failed to fetch leadership team", res.status);
      return [];
    }

    const data = await res.json();
    if (data.success && data.data) {
      return data.data as Leader[];
    }
    return [];
  } catch (err) {
    console.error("Error fetching leadership team", err);
    return [];
  }
}

export default async function CommiteePage() {
  const leaders = await getLeaders();

  // Director message → first leader with is_active_message = 1
  const director = leaders.find((l) => l.is_active_message === 1);

  // Rest of committee members (exclude director if already used)
  const members = leaders.filter((l) => l.is_active_message !== 1);

  return (
    <>
      <Banner
        title="Commitee"
        breadcrumbs={[
          { label: "Home", href: "/" },
          { label: "About", href: "/about" },
          { label: "Commitee" },
        ]}
      />

      {/* Director Message Section */}
      {director && (
        <section className="pt-[80px] sm:pt-[100px] lg:pt-[130px] bg-white hidden">
          <div className="container px-4 mx-auto">
            <div className="flex flex-col md:flex-row items-center gap-10  md:gap-10 lg:gap-16">
              {/* Left Image */}
              <div className="relative flex-1 w-full">
                <Image
                  src={director.image_url}
                  alt={director.full_name}
                  width={700}
                  height={700}
                  className="rounded-[20px] sm:rounded-[25px] lg:rounded-[30px] object-cover w-full h-[300px] sm:h-[400px] md:h-[500px]"
                />
              </div>

              {/* Right Content */}
              <div className="flex-1 text-center md:text-left">
                <span className="sub-header font-semibold">
                  Director Message
                </span>

                <h2 className="text-3xl sm:text-5xl lg:text-7xl font-bold font-philosopher mb-4 leading-tight">
                  {director.full_name}
                </h2>

                <p className="text-[#909090] leading-relaxed font-cabin text-sm sm:text-base mb-4 max-w-[700px] mx-auto md:mx-0">
                  {director.message}
                </p>

                <p className="text-base sm:text-lg font-bold mt-2">
                  - {director.designation}
                </p>
              </div>
            </div>
          </div>
        </section>
      )}

      {/* Team Section */}
      <section className="py-[80px] sm:py-[100px] lg:py-[130px] bg-white">
        <div className="container mx-auto px-4 text-center">
          <span className="sub-header font-semibold">Executives</span>
          <h2 className="text-4xl text-center sm:text-6xl lg:text-7xl font-bold mb-5 md:mb-5">
            Executive Committee
          </h2>
          <p className="text-[#909090] leading-relaxed font-cabin text-sm sm:text-base md:text-lg max-w-3xl mx-auto px-2 sm:px-6 md:px-10 mb-30 md:mb-30">
            Pussalla is spearheaded by a visionary team of leaders who have been
            the essence of our vertically integrated farm-to-fork business
            model, while enforcing a socially responsible operation.
          </p>

          {/* Commitee Members Grid */}
          <div className="grid gap-x-5 gap-y-10 sm:grid-cols-1 md:grid-cols-2 lg:grid-cols-4">
            {members.map((item, index) => (
              <CommiteeCard
                key={index}
                img={item.image_url}
                designation={item.designation}
                name={item.full_name}
              />
            ))}
          </div>
        </div>
      </section>
    </>
  );
}
