import Banner from "@/components/Banner";
import AwardsFilter from "@/components/YearFilter";
import AwardCard from "@/components/AwardCard";

export const dynamic = "force-dynamic"; // <- ADD THIS HERE

type Award = {
  title: string;
  description: string;
  year: string;
  image_url: string;
};

type Certificate = {
  title: string;
  description: string;
  year: string;
  image_url: string;
};

async function getAwards() {
  const res = await fetch(
    `${process.env.NEXT_PUBLIC_BASE_URL}/api/v1/awards/active`,
    { cache: "no-store" } // ensures fresh data (SSR)
  );
  if (!res.ok) throw new Error("Failed to fetch awards");
  const json = await res.json();
  return json.data as Record<string, Award[]>;
}

async function getCertificates() {
  const res = await fetch(
    `${process.env.NEXT_PUBLIC_BASE_URL}/api/v1/certificates/active`,
    {
      next: { revalidate: 0 }, // SSR fresh data
    }
  );

  if (!res.ok) throw new Error("Failed to fetch certificates");

  const data = await res.json();
  return data.data as Certificate[];
}

export default async function AwardsPage() {
  const awardsByYear = await getAwards();
  const certificates = await getCertificates();

  // Flatten awards into an array with year property
  const allAwards = Object.entries(awardsByYear).flatMap(([year, list]) =>
    list.map((award) => ({
      ...award,
      year,
    }))
  );

  const years = Object.keys(awardsByYear)
    .map((y) => parseInt(y, 10))
    .sort((a, b) => b - a);

  return (
    <>
      <Banner
        title="Awards & Certifications"
        breadcrumbs={[
          { label: "Home", href: "/" },
          { label: "Awards & Certifications" },
        ]}
      />

      {/* Awards Section */}
      <section className="py-[80px] sm:py-[100px] lg:py-[130px] bg-white">
        <div className="container mx-auto px-4">
          <h2 className="text-4xl text-center sm:text-6xl lg:text-7xl xl:text-6xl font-bold mb-5">
            Awards
          </h2>
          <p className="text-[#909090] text-center leading-relaxed font-cabin text-sm sm:text-base md:text-lg max-w-3xl mx-auto px-2 sm:px-6 md:px-10 mb-15">
            Throughout our journey, we have garnered national and international
            recognition through awards from reputable local and international
            institutional bodies. These are testaments to our excellence in
            quality, innovation, and business processes, which have been
            realized with genuine commitment rather than being mere business
            statements.
          </p>
          {/* Year Filter (Client Component) */}
          <AwardsFilter years={years} awards={allAwards} />
        </div>
      </section>

      {/* Certificates Section */}
      <section className="pb-[80px] sm:pb-[100px] lg:pb-[130px] bg-white">
        <div className="container mx-auto px-4">
          <h2 className="text-4xl text-center sm:text-6xl lg:text-7xl xl:text-6xl font-bold mt-2 mb-10">
            Certificates
          </h2>

          <div className="grid gap-x-5 gap-y-10 sm:grid-cols-1 md:grid-cols-2 lg:grid-cols-4">
            {certificates.map((item, index) => (
              <AwardCard
                key={index}
                img={item.image_url}
                title={item.title}
                year={item.year}
                awardingBody={item.description}
              />
            ))}
          </div>
        </div>
      </section>
    </>
  );
}
